jQuery(document).ready(function($) {
    // Wait for editors to be initialized
    if (typeof tinymce !== 'undefined') {
        tinymce.on('AddEditor', function(e) {
            setTimeout(function() {
                if (e.editor.id === 'content') {
                    addMainDescriptionButton();
                } else if (e.editor.id === 'excerpt') {
                    addShortDescriptionButton();
                }
            }, 500);
        });
    }

    // Store generated descriptions
    let generatedDescriptions = {
        main: '',
        short: ''
    };

    // Store generated brands and tags
    let generatedBrands = [];
    let generatedTags = [];

    // Store initialization state
    const initialized = {
        main: false,
        short: false,
        brands: false,
        tags: false
    };

    // Add loading indicator HTML using WordPress spinner for both generators
    $('body').append(`
        <div id="main-description-generating" class="description-generating" style="display: none;">
            <span class="spinner is-active"></span>
            <span>Generating product description...</span>
        </div>
        <div id="short-description-generating" class="description-generating" style="display: none;">
            <span class="spinner is-active"></span>
            <span>Generating short description...</span>
        </div>
        <div id="brands-generating" class="description-generating" style="display: none;">
            <span class="spinner is-active"></span>
            <span>Generating brands...</span>
        </div>
        <div id="tags-generating" class="description-generating" style="display: none;">
            <span class="spinner is-active"></span>
            <span>Generating tags...</span>
        </div>
    `);

    // Generate description function
    function generateDescription(type) {
        const postId = wcProdDescGen.postId;
        const tone = $(`#${type}-description-tone`).val();
        const previewContainer = $(`#${type}-description-preview-container`);
        const previewContent = $(`#${type}-description-preview-content`);
        const loadingDiv = $(`#${type}-description-generating`);
        const editorContainer = $(`#wp-${type === 'main' ? 'content' : 'excerpt'}-wrap`);

        if (!previewContainer.length || !previewContent.length) return;

        // Hide preview and show loading
        previewContainer.hide();
        loadingDiv.insertAfter(editorContainer).show();

        $.ajax({
            url: wcProdDescGen.ajaxurl,
            type: 'POST',
            data: {
                action: type === 'main' ? 'generate_product_description' : 'generate_product_short_description',
                post_id: postId,
                nonce: wcProdDescGen.nonce,
                tone: tone
            },
            success: function(response) {
                // Hide loading
                loadingDiv.hide();

                if (response.success) {
                    generatedDescriptions[type] = response.data.description;
                    previewContent.html(response.data.description);
                    previewContainer.show();
                    previewContainer.find('button').show();
                } else {
                    alert(response.data.message || 'Failed to generate description');
                }
            },
            error: function(xhr, status, error) {
                // Hide loading
                loadingDiv.hide();
                alert('Error: ' + error);
            }
        });
    }

    // Apply description function
    function applyDescription(type) {
        const description = generatedDescriptions[type];
        if (!description) return;

        const editor = tinymce.get(type === 'main' ? 'content' : 'excerpt');
        if (editor) {
            editor.setContent(description);
        } else {
            // Fallback for text mode
            const textarea = $(`#${type === 'main' ? 'content' : 'excerpt'}`);
            textarea.val(description);
        }

        // Hide preview container
        $(`#${type}-description-preview-container`).hide();
    }

    // Generate brands function
    function generateBrands() {
        const postId = wcProdDescGen.postId;
        const previewContainer = $('#brands-preview-container');
        const previewContent = $('#brands-preview-content');
        const loadingDiv = $('#brands-generating');
        const brandsContainer = $('#taxonomy-product_brand');

        if (!previewContainer.length || !previewContent.length) return;

        // Hide preview and show loading
        previewContainer.hide();
        loadingDiv.insertAfter(brandsContainer).show();

        $.ajax({
            url: wcProdDescGen.ajaxurl,
            type: 'POST',
            data: {
                action: 'generate_product_brands',
                post_id: postId,
                nonce: wcProdDescGen.nonce
            },
            success: function(response) {
                // Hide loading
                loadingDiv.hide();

                if (response.success) {
                    generatedBrands = response.data.brands;
                    previewContent.html(generatedBrands.map(brand => 
                        `<div class="brand-item"><input type="checkbox" value="${brand}" checked> ${brand}</div>`
                    ).join(''));
                    previewContainer.show();
                    previewContainer.find('button').show();
                } else {
                    alert(response.data.message || 'Failed to generate brands');
                }
            },
            error: function(xhr, status, error) {
                // Hide loading
                loadingDiv.hide();
                alert('Error: ' + error);
            }
        });
    }

    // Apply brands function
    function applyBrands() {
        if (!generatedBrands.length) return;

        const selectedBrands = [];
        $('#brands-preview-content input:checked').each(function() {
            selectedBrands.push($(this).val());
        });

        console.log('Selected brands:', selectedBrands);

        if (selectedBrands.length > 0) {
            // Send AJAX request to create brands and save to product
            $.ajax({
                url: wcProdDescGen.ajaxurl,
                type: 'POST',
                data: {
                    action: 'apply_product_brands',
                    brands: selectedBrands,
                    post_id: wcProdDescGen.postId,
                    nonce: wcProdDescGen.nonce
                },
                success: function(response) {
                    if (response.success) {
                        alert('Brands applied successfully!');
                        
                        // Refresh the brand field if it exists
                        const brandField = $('#taxonomy-product_brand');
                        if (brandField.length && response.data.brand_ids) {
                            // Update the select field with the new brand IDs
                            response.data.brand_ids.forEach(function(brandId) {
                                const existingOption = brandField.find(`option[value="${brandId}"]`);
                                if (existingOption.length === 0) {
                                    // Find the brand name from the response
                                    const brandName = response.data.brand_names[brandId] || '';
                                    if (brandName) {
                                        const newOption = $('<option></option>')
                                            .val(brandId)
                                            .text(brandName)
                                            .prop('selected', true);
                                        brandField.append(newOption);
                                    }
                                } else {
                                    existingOption.prop('selected', true);
                                }
                            });
                        }
                    } else {
                        alert('Error: ' + (response.data.message || 'Failed to apply brands'));
                    }
                },
                error: function(xhr, status, error) {
                    alert('Error: ' + error);
                }
            });
        } else {
            alert('Please select at least one brand to apply.');
        }

        // Hide preview container
        $('#brands-preview-container').hide();
    }

    // Generate tags function
    function generateTags() {
        const postId = wcProdDescGen.postId;
        const previewContainer = $('#tags-preview-container');
        const previewContent = $('#tags-preview-content');
        const loadingDiv = $('#tags-generating');
        const tagsContainer = $('#product_tag');

        if (!previewContainer.length || !previewContent.length) return;

        // Hide preview and show loading
        previewContainer.hide();
        loadingDiv.insertAfter(tagsContainer).show();

        $.ajax({
            url: wcProdDescGen.ajaxurl,
            type: 'POST',
            data: {
                action: 'generate_product_tags',
                post_id: postId,
                nonce: wcProdDescGen.nonce
            },
            success: function(response) {
                // Hide loading
                loadingDiv.hide();

                if (response.success) {
                    generatedTags = response.data.tags;
                    previewContent.html(generatedTags.map(tag => 
                        `<div class="tag-item"><input type="checkbox" value="${tag}" checked> ${tag}</div>`
                    ).join(''));
                    previewContainer.show();
                    previewContainer.find('button').show();
                } else {
                    alert(response.data.message || 'Failed to generate tags');
                }
            },
            error: function(xhr, status, error) {
                // Hide loading
                loadingDiv.hide();
                alert('Error: ' + error);
            }
        });
    }

    // Apply tags function
    function applyTags() {
        if (!generatedTags.length) return;

        const selectedTags = [];
        $('#tags-preview-content input:checked').each(function() {
            selectedTags.push($(this).val());
        });

        console.log('Selected tags:', selectedTags);

        if (selectedTags.length > 0) {
            // Send AJAX request to create tags and save to product
            $.ajax({
                url: wcProdDescGen.ajaxurl,
                type: 'POST',
                data: {
                    action: 'apply_product_tags',
                    tags: selectedTags,
                    post_id: wcProdDescGen.postId,
                    nonce: wcProdDescGen.nonce
                },
                success: function(response) {
                    if (response.success) {
                        alert('Tags applied successfully!');
                        
                        // Refresh the tag field if it exists
                        const tagField = $('#product_tag');
                        if (tagField.length && response.data.tag_ids) {
                            // Update the select field with the new tag IDs
                            response.data.tag_ids.forEach(function(tagId) {
                                const existingOption = tagField.find(`option[value="${tagId}"]`);
                                if (existingOption.length === 0) {
                                    // Find the tag name from the response
                                    const tagName = response.data.tag_names[tagId] || '';
                                    if (tagName) {
                                        const newOption = $('<option></option>')
                                            .val(tagId)
                                            .text(tagName)
                                            .prop('selected', true);
                                        tagField.append(newOption);
                                    }
                                } else {
                                    existingOption.prop('selected', true);
                                }
                            });
                        }
                    } else {
                        alert('Error: ' + (response.data.message || 'Failed to apply tags'));
                    }
                },
                error: function(xhr, status, error) {
                    alert('Error: ' + error);
                }
            });
        } else {
            alert('Please select at least one tag to apply.');
        }

        // Hide preview container
        $('#tags-preview-container').hide();
    }

    // Add function to add main description button
    function addMainDescriptionButton() {
        const $editorTools = $('#wp-content-media-buttons');
        if (!$editorTools.length || initialized.main) return;

        const $container = $('<span class="description-generator-container"></span>');
        const $toneSelector = createToneSelector('main-description-tone');
        const $button = $('<button type="button" class="button">Generate Description</button>')
            .on('click', function() { generateDescription('main'); });

        $container.append($toneSelector, $button);
        $editorTools.append($container);

        addPreviewContainer('main');
        initialized.main = true;
    }

    // Add function to add short description button
    function addShortDescriptionButton() {
        const $editorTools = $('#wp-excerpt-media-buttons');
        if (!$editorTools.length || initialized.short) return;

        const $container = $('<span class="description-generator-container"></span>');
        const $toneSelector = createToneSelector('short-description-tone');
        const $button = $('<button type="button" class="button">Generate Short Description</button>')
            .on('click', function() { generateDescription('short'); });

        $container.append($toneSelector, $button);
        $editorTools.append($container);

        addPreviewContainer('short');
        initialized.short = true;
    }

    // Add function to create tone selector
    function createToneSelector(id) {
        return $('<select></select>')
            .attr('id', id)
            .addClass('components-select-control__input')
            .append(
                $('<option value="formal">Formal & Professional</option>'),
                $('<option value="casual">Casual & Friendly</option>'),
                $('<option value="persuasive">Persuasive & Sales-focused</option>'),
                $('<option value="technical">Technical & Detailed</option>'),
                $('<option value="luxury">Luxury & Premium</option>'),
                $('<option value="creative">Creative & Engaging</option>')
            );
    }

    // Add function to create preview container
    function addPreviewContainer(type) {
        const $editorTools = $(`#wp-${type === 'main' ? 'content' : 'excerpt'}-editor-tools`);
        if (!$editorTools.length) return;

        const $previewContainer = $('<div></div>')
            .attr('id', `${type}-description-preview-container`)
            .addClass('description-preview-container');

        const $previewContent = $('<div></div>')
            .attr('id', `${type}-description-preview-content`);

        const $actionContainer = $('<div></div>')
            .addClass('description-action-container');

        const $applyButton = $('<button type="button" class="button button-primary">Apply Description</button>')
            .on('click', function() { applyDescription(type); });

        const $regenerateButton = $('<button type="button" class="button">Generate Another</button>')
            .on('click', function() { generateDescription(type); });

        $actionContainer.append($applyButton, $regenerateButton);
        $previewContainer.append($previewContent, $actionContainer);
        $editorTools.after($previewContainer);
    }

    // Add function to add brands button
    function addBrandsButton() {
        // Try multiple possible field selectors for brands
        const $brandsContainer = $('#taxonomy-product_brand');
        console.log('Looking for brand fields:', $brandsContainer.length, $brandsContainer.map(function() { return this.id || this.name; }).get());
        
        if (!$brandsContainer.length || initialized.brands) return;

        const $container = $('<div class="brands-generator-container"></div>');
        const $button = $('<button type="button" class="button">' + wcProdDescGen.generateBrands + '</button>')
            .on('click', function() { generateBrands(); });

        $container.append($button);
        $brandsContainer.first().after($container);

        addBrandsPreviewContainer();
        initialized.brands = true;
    }

    // Add function to add brands preview container
    function addBrandsPreviewContainer() {
        // Try multiple possible field selectors for brands
        const $brandsContainer = $('#taxonomy-product_brand');
        if (!$brandsContainer.length) return;

        const $previewContainer = $('<div></div>')
            .attr('id', 'brands-preview-container')
            .addClass('description-preview-container');

        const $previewContent = $('<div></div>')
            .attr('id', 'brands-preview-content');

        const $actionContainer = $('<div></div>')
            .addClass('description-action-container');

        const $applyButton = $('<button type="button" class="button button-primary">' + wcProdDescGen.applyBrands + '</button>')
            .on('click', function() { applyBrands(); });

        const $regenerateButton = $('<button type="button" class="button">' + wcProdDescGen.generateMoreBrands + '</button>')
            .on('click', function() { generateBrands(); });

        $actionContainer.append($applyButton, $regenerateButton);
        $previewContainer.append($previewContent, $actionContainer);
        $brandsContainer.first().after($previewContainer);
    }

    // Add function to add tags button
    function addTagsButton() {
        // Try multiple possible field selectors for tags
        const $tagsContainer = $('#product_tag');
        console.log('Looking for tag fields:', $tagsContainer.length, $tagsContainer.map(function() { return this.id || this.name; }).get());
        
        if (!$tagsContainer.length || initialized.tags) return;

        const $container = $('<div class="tags-generator-container"></div>');
        const $button = $('<button type="button" class="button">' + wcProdDescGen.generateTags + '</button>')
            .on('click', function() { generateTags(); });

        $container.append($button);
        $tagsContainer.first().after($container);

        addTagsPreviewContainer();
        initialized.tags = true;
    }

    // Add function to add tags preview container
    function addTagsPreviewContainer() {
        // Try multiple possible field selectors for tags
        const $tagsContainer = $('#product_tag');
        if (!$tagsContainer.length) return;

        const $previewContainer = $('<div></div>')
            .attr('id', 'tags-preview-container')
            .addClass('description-preview-container');

        const $previewContent = $('<div></div>')
            .attr('id', 'tags-preview-content');

        const $actionContainer = $('<div></div>')
            .addClass('description-action-container');

        const $applyButton = $('<button type="button" class="button button-primary">' + wcProdDescGen.applyTags + '</button>')
            .on('click', function() { applyTags(); });

        const $regenerateButton = $('<button type="button" class="button">' + wcProdDescGen.generateMoreTags + '</button>')
            .on('click', function() { generateTags(); });

        $actionContainer.append($applyButton, $regenerateButton);
        $previewContainer.append($previewContent, $actionContainer);
        $tagsContainer.first().after($previewContainer);
    }

    // Initialize description buttons when document is ready
    $(document).ready(function() {
        // Remove any existing buttons first
        $('.description-generator-container').remove();
        $('.description-preview-container').remove();
        $('.brands-generator-container').remove();
        $('.tags-generator-container').remove();
        
        // Reset initialization state
        initialized.main = false;
        initialized.short = false;
        initialized.brands = false;
        initialized.tags = false;

        // Add buttons
        addMainDescriptionButton();
        addShortDescriptionButton();
        addBrandsButton();
        addTagsButton();
    });

    // Also handle TinyMCE editor initialization
    if (typeof tinymce !== 'undefined') {
        tinymce.on('AddEditor', function(e) {
            if (e.editor.id === 'content') {
                addMainDescriptionButton();
            } else if (e.editor.id === 'excerpt') {
                addShortDescriptionButton();
            }
        });
    }

    // Add this function to handle image generation
    function generateProductImage(postId) {
        const loadingDiv = $('#featured-image-generation-status');
        const previewContainer = $('#featured-image-preview-container');
        const imageGrid = $('#featured-image-grid');

        loadingDiv.html('<span class="spinner is-active"></span><span>Generating images...</span>').show();
        previewContainer.hide();

        $.ajax({
            url: wcProdDescGen.ajaxurl,
            type: 'POST',
            data: {
                action: 'generate_product_image',
                post_id: postId,
                nonce: wcProdDescGen.nonce,
                preview_only: '1'
            },
            success: function(response) {
                // Hide loading div after successful generation
                loadingDiv.hide();

                if (response.success && response.data.image_urls) {
                    imageGrid.empty();
                    response.data.image_urls.forEach(function(url) {
                        const wrapper = $('<div></div>').css({
                            position: 'relative',
                            border: '1px solid #ddd',
                            padding: '5px'
                        });

                        const img = $('<img>').attr('src', url).css({
                            width: '100%',
                            display: 'block'
                        });

                        const useButton = $('<button></button>')
                            .addClass('button button-primary use-as-featured')
                            .text('Use as Featured')
                            .css({
                                margin: '5px 0 0 0',
                                width: '100%'
                            })
                            .attr('data-image-url', url)
                            .attr('data-post-id', postId);

                        wrapper.append(img, useButton);
                        imageGrid.append(wrapper);
                    });
                    previewContainer.show();
                } else {
                    // Show error message if generation failed
                    loadingDiv.html(response.message || 'Failed to generate images').show();
                    setTimeout(() => loadingDiv.fadeOut(), 3000);
                }
            },
            error: function(xhr, status, error) {
                // Show error message and hide after delay
                loadingDiv.html('Error: ' + error).show();
                setTimeout(() => loadingDiv.fadeOut(), 3000);
            }
        });
    }

    // Add this function to handle successful image attachment
    function handleImageAttachSuccess(response, imageUrl, isGallery) {
        if (isGallery) {
            // Gallery image handling...
        } else {
            // Update featured image preview
            const previewContainer = $('#featured-image-preview-container');
            const imageGrid = $('#featured-image-grid');
            
            // Create success message
            const successMessage = $('<div class="notice notice-success"></div>')
                .text('Featured image set successfully')
                .css('margin', '10px 0');
            
            // Create image container
            const imageContainer = $('<div></div>')
                .addClass('featured-image-container')
                .css({
                    'border': '1px solid #ddd',
                    'padding': '10px',
                    'margin-top': '10px'
                });
            
            // Add image
            const img = $('<img>')
                .attr('src', imageUrl)
                .css({
                    'width': '100%',
                    'height': 'auto',
                    'display': 'block'
                });
            
            imageContainer.append(img);
            
            // Clear and update the preview
            imageGrid.empty()
                .append(successMessage)
                .append(imageContainer);
            
            previewContainer.show();
            
            // Update the WP featured image area if it exists
            const wpFeaturedImage = $('#postimagediv');
            if (wpFeaturedImage.length) {
                // Trigger WP's featured image update
                if (typeof wp !== 'undefined' && wp.media && wp.media.featuredImage) {
                    wp.media.featuredImage.set(response.data.attachment_id);
                }
            }
        }
    }

    // Update the attachImage function
    function attachImage(imageUrl, postId, isGallery = false) {
        const loadingDiv = isGallery ? 
            $('#gallery-generation-status') : 
            $('#featured-image-generation-status');
        const previewContainer = isGallery ? 
            $('#gallery-image-preview-container') : 
            $('#featured-image-preview-container');

        loadingDiv.html('<span class="spinner is-active"></span><span>Attaching image...</span>').show();

        const data = new FormData();
        data.append('action', 'attach_generated_image');
        data.append('nonce', wcProdDescGen.nonce);
        data.append('post_id', postId);
        data.append('image_url', imageUrl);
        data.append('is_gallery', isGallery ? '1' : '0');

        jQuery.ajax({
            url: wcProdDescGen.ajaxurl,
            type: 'POST',
            data: data,
            processData: false,
            contentType: false,
            success: function(response) {
                if (response.success) {
                    loadingDiv.html(`<div class="notice notice-success">${response.data.message}</div>`);
                    
                    // Update UI with new image
                    handleImageAttachSuccess(response, imageUrl, isGallery);
                    
                    // Update WordPress media frame if it exists
                    if (wp.media && wp.media.frame) {
                        wp.media.frame.setState('library').get('selection').reset();
                        wp.media.frame.content.get().collection.props.set({ignore: (+ new Date())});
                        wp.media.frame.content.get().options.selection.reset();
                    }
                } else {
                    loadingDiv.html(`<div class="notice notice-error">${response.data.message}</div>`);
                }
                previewContainer.show();
            },
            error: function(xhr, status, error) {
                loadingDiv.html(`<div class="notice notice-error">Error: ${error}</div>`);
                previewContainer.show();
            }
        });
    }

    // Update the click handler
    $(document).on('click', '.use-as-featured', function(e) {
        e.preventDefault();
        e.stopPropagation(); // Prevent event bubbling
        
        const imageUrl = $(this).attr('data-image-url');
        const postId = $(this).attr('data-post-id');
        
        if (!imageUrl || !postId) {
            console.error('Missing image URL or post ID');
            return;
        }
        
        // Disable the button while processing
        $(this).prop('disabled', true).text('Setting...');
        
        attachImage(imageUrl, postId, false);
    });

    // Update the click handler for "Add to gallery" buttons
    jQuery(document).on('click', '.add-to-gallery', function(e) {
        e.preventDefault();
        const imageUrl = jQuery(this).data('image-url');
        const postId = jQuery(this).data('post-id');
        attachImage(imageUrl, postId, true);
    });

    // Add event listener for the Generate Featured Image button
    $(document).on('click', '#generate-product-image', function() {
        const postId = $(this).data('post-id');
        generateProductImage(postId);
    });

    // Add event listener for the Generate Gallery Images button
    $(document).on('click', '#generate-gallery-images', function() {
        const postId = $(this).data('post-id');
        generateGalleryImages(postId);
    });

    // Update the generateGalleryImages function to handle multiple images properly
    function generateGalleryImages(postId) {
        const loadingDiv = $('#gallery-generation-status');
        const previewContainer = $('#gallery-image-preview-container');
        const imageGrid = $('#gallery-image-grid');

        loadingDiv.html('<span class="spinner is-active"></span><span>Generating gallery images...</span>').show();
        previewContainer.hide();

        $.ajax({
            url: wcProdDescGen.ajaxurl,
            type: 'POST',
            data: {
                action: 'generate_product_image',
                post_id: postId,
                nonce: wcProdDescGen.nonce,
                preview_only: '1',
                is_gallery: '1'
            },
            success: function(response) {
                loadingDiv.hide();

                if (response.success && response.data.image_urls) {
                    imageGrid.empty();

                    // Add select all container
                    const selectAllContainer = $('<div></div>')
                        .addClass('select-all-container')
                        .css({
                            'margin-bottom': '10px',
                            'padding': '10px',
                            'background': '#f9f9f9',
                            'border': '1px solid #ddd'
                        });

                    const selectAllCheckbox = $('<input type="checkbox" id="select-all-images">')
                        .on('change', function() {
                            $('.gallery-image-checkbox').prop('checked', this.checked);
                        });

                    selectAllContainer.append(
                        selectAllCheckbox,
                        $('<label for="select-all-images">').text(' Select All Images')
                    );

                    // Add "Add Selected to Gallery" button
                    const addSelectedButton = $('<button></button>')
                        .addClass('button button-primary')
                        .text('Add Selected to Gallery')
                        .css('margin-left', '10px')
                        .on('click', function(e) {
                            e.preventDefault();
                            const selectedUrls = [];
                            $('.gallery-image-checkbox:checked').each(function() {
                                selectedUrls.push($(this).attr('data-url'));
                            });
                            if (selectedUrls.length > 0) {
                                attachMultipleGalleryImages(selectedUrls, postId);
                            }
                        });

                    selectAllContainer.append(addSelectedButton);
                    imageGrid.before(selectAllContainer);

                    // Generate image grid with checkboxes
                    response.data.image_urls.forEach(function(url) {
                        const wrapper = $('<div></div>').css({
                            position: 'relative',
                            border: '1px solid #ddd',
                            padding: '5px'
                        });

                        const checkbox = $('<input type="checkbox">')
                            .addClass('gallery-image-checkbox')
                            .attr('data-url', url)
                            .css({
                                position: 'absolute',
                                top: '10px',
                                left: '10px',
                                zIndex: '1'
                            });

                        const img = $('<img>').attr('src', url).css({
                            width: '100%',
                            display: 'block'
                        });

                        wrapper.append(checkbox, img);
                        imageGrid.append(wrapper);
                    });
                    previewContainer.show();
                } else {
                    loadingDiv.html(response.message || 'Failed to generate images').show();
                }
            },
            error: function(xhr, status, error) {
                loadingDiv.html('Error: ' + error).show();
            }
        });
    }

    // Update the attachMultipleGalleryImages function
    function attachMultipleGalleryImages(imageUrls, postId) {
        const loadingDiv = $('#gallery-generation-status');
        const previewContainer = $('#gallery-image-preview-container');
        const imageGrid = $('#gallery-image-grid');
        
        loadingDiv.html('<span class="spinner is-active"></span><span>Adding images to gallery...</span>').show();

        // First, get existing gallery images
        $.ajax({
            url: wcProdDescGen.ajaxurl,
            type: 'POST',
            data: {
                action: 'get_product_gallery',
                post_id: postId,
                nonce: wcProdDescGen.nonce
            },
            success: function(existingGallery) {
                let completedUploads = 0;
                const totalUploads = imageUrls.length;
                const successfulUploads = [];
                const existingIds = existingGallery.data || [];

                imageUrls.forEach(function(imageUrl) {
                    const data = new FormData();
                    data.append('action', 'attach_generated_image');
                    data.append('nonce', wcProdDescGen.nonce);
                    data.append('post_id', postId);
                    data.append('image_url', imageUrl);
                    data.append('is_gallery', '1');
                    data.append('existing_gallery', JSON.stringify(existingIds));

                    $.ajax({
                        url: wcProdDescGen.ajaxurl,
                        type: 'POST',
                        data: data,
                        processData: false,
                        contentType: false,
                        success: function(response) {
                            completedUploads++;
                            if (response.success) {
                                successfulUploads.push({
                                    url: imageUrl,
                                    id: response.data.attachment_id
                                });
                            }
                            
                            if (completedUploads === totalUploads) {
                                // Show success message
                                loadingDiv.html(`<div class="notice notice-success">
                                    <p>${successfulUploads.length} of ${totalUploads} images added to gallery successfully!</p>
                                    <p>Refreshing page to show updated gallery...</p>
                                </div>`);
                                
                                // Update the grid with success indicators
                                $('.gallery-image-checkbox:checked').each(function() {
                                    const checkbox = $(this);
                                    const wrapper = checkbox.closest('div');
                                    checkbox.prop('disabled', true);
                                    
                                    $('<div></div>')
                                        .addClass('image-status')
                                        .css({
                                            'text-align': 'center',
                                            'margin-top': '5px',
                                            'color': '#1e8cbe'
                                        })
                                        .text('Added to gallery')
                                        .appendTo(wrapper);
                                });

                                // Reload the page after a short delay to show the updated gallery
                                setTimeout(function() {
                                    window.location.reload();
                                }, 2000);
                            }
                        },
                        error: function(xhr, status, error) {
                            completedUploads++;
                            if (completedUploads === totalUploads) {
                                loadingDiv.html(`<div class="notice notice-error">
                                    <p>Some images failed to upload. Error: ${error}</p>
                                </div>`);
                            }
                        }
                    });
                });
            },
            error: function(xhr, status, error) {
                loadingDiv.html(`<div class="notice notice-error">
                    <p>Failed to get existing gallery. Error: ${error}</p>
                </div>`);
            }
        });
    }
}); 